<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../includes/config.php';

header('Content-Type: application/json');

// Start session and check authentication
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// Params: account_id (required), from (date, optional), to (date, optional)
$accountId = isset($_GET['account_id']) ? intval($_GET['account_id']) : 0;
$fromDate = isset($_GET['from']) ? $_GET['from'] : null;
$toDate = isset($_GET['to']) ? $_GET['to'] : null;
$currency = isset($_GET['currency']) ? $_GET['currency'] : null; // optional override

if ($accountId <= 0) {
    http_response_code(400);
    echo json_encode(['error' => 'account_id is required']);
    exit;
}

try {
    global $db1;

    // Get account
    $stmtAcc = $db1->prepare("SELECT id, name, type, currency FROM accounts WHERE id = :id");
    $stmtAcc->execute([':id' => $accountId]);
    $account = $stmtAcc->fetch(PDO::FETCH_ASSOC);
    if (!$account) {
        throw new Exception('Account not found');
    }

    // Currency logic: prefer explicit query currency, then account.currency, else report all
    $currencyFilter = $currency ?: ($account['currency'] ?: null);

    // Opening balance up to fromDate-1
    $opening = 0.0;
    if ($fromDate) {
        $sqlOpening = "
            SELECT COALESCE(SUM(CASE WHEN type = 'credit' THEN amount ELSE -amount END), 0) AS bal
            FROM account_ledger
            WHERE account_id = :aid
              AND entry_date < :fromDate
              " . ($currencyFilter ? "AND currency = :cur" : "") . "
        ";
        $stmtOpen = $db1->prepare($sqlOpening);
        $stmtOpen->bindValue(':aid', $accountId, PDO::PARAM_INT);
        $stmtOpen->bindValue(':fromDate', $fromDate);
        if ($currencyFilter) $stmtOpen->bindValue(':cur', $currencyFilter);
        $stmtOpen->execute();
        $opening = (float)($stmtOpen->fetchColumn() ?: 0);
    }

    // Statement rows within range
    $sqlRows = "
        SELECT id, entry_date, type, amount, currency, source, source_id, description
        FROM account_ledger
        WHERE account_id = :aid
          " . ($fromDate ? "AND entry_date >= :fromDate" : "") . "
          " . ($toDate ? "AND entry_date <= :toDate" : "") . "
          " . ($currencyFilter ? "AND currency = :cur" : "") . "
        ORDER BY entry_date ASC, id ASC
    ";
    $stmtRows = $db1->prepare($sqlRows);
    $stmtRows->bindValue(':aid', $accountId, PDO::PARAM_INT);
    if ($fromDate) $stmtRows->bindValue(':fromDate', $fromDate);
    if ($toDate) $stmtRows->bindValue(':toDate', $toDate);
    if ($currencyFilter) $stmtRows->bindValue(':cur', $currencyFilter);
    $stmtRows->execute();
    $rows = $stmtRows->fetchAll(PDO::FETCH_ASSOC);

    // Running balance
    $running = $opening;
    foreach ($rows as &$r) {
        $running += ($r['type'] === 'credit' ? (float)$r['amount'] : -(float)$r['amount']);
        $r['running_balance'] = $running;
    }
    unset($r);

    echo json_encode([
        'success' => true,
        'account' => $account,
        'currency' => $currencyFilter,
        'opening_balance' => $opening,
        'closing_balance' => $running,
        'rows' => $rows
    ]);
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode(['error' => $e->getMessage()]);
}
?>


