<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Set content type to JSON
header('Content-Type: application/json');

// Include database connection
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/DataBaseObjects.php';

// Check if user is logged in
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

// Check if it's a POST request
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

try {
    // Get form data
    $invoice_id = $_POST['invoice_id'] ?? 0;
    $client_id = $_POST['client_id'] ?? 0;
    $invoice_number = $_POST['invoice_number'] ?? '';
    $issue_date = $_POST['issue_date'] ?? '';
    $due_date = $_POST['due_date'] ?? '';
    $currency = $_POST['currency'] ?? '';
    $notes = $_POST['notes'] ?? '';
    $items = $_POST['items'] ?? [];
    $issue_invoice = isset($_POST['issue_invoice']) && $_POST['issue_invoice'] == '1';
    
    // Validate required fields
    if (!$invoice_id || !$client_id || !$invoice_number || !$issue_date || !$currency || empty($items)) {
        throw new Exception('All required fields must be filled');
    }
    
    // Start transaction
    $db = new DataBaseObjects();
    $db1->beginTransaction();
    
    // Check if invoice exists and is editable
    $stmt = $db1->prepare("SELECT status FROM invoices WHERE id = ?");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$invoice) {
        throw new Exception('Invoice not found');
    }
    
    if ($invoice['status'] !== 'draft') {
        throw new Exception('Only draft invoices can be edited');
    }
    
    // Update invoice header
    $stmt = $db1->prepare("
        UPDATE invoices 
        SET client_id = ?, invoice_number = ?, issue_date = ?, due_date = ?, 
            currency = ?, notes = ?, updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
    ");
    $stmt->execute([$client_id, $invoice_number, $issue_date, $due_date, $currency, $notes, $invoice_id]);
    
    // Delete existing invoice items
    $stmt = $db1->prepare("DELETE FROM invoice_items WHERE invoice_id = ?");
    $stmt->execute([$invoice_id]);
    
    // Delete existing credits ledger entries for this invoice
    $stmt = $db1->prepare("DELETE FROM credits_ledger WHERE invoice_id = ?");
    $stmt->execute([$invoice_id]);
    
    // Calculate totals
    $subtotal = 0;
    $total = 0;
    
    // Insert new invoice items
    foreach ($items as $item) {
        if (empty($item['application_id']) || empty($item['description']) || 
            empty($item['credit_units']) || empty($item['unit_price'])) {
            continue;
        }
        
        $quantity = floatval($item['quantity'] ?? 1);
        $unit_price = floatval($item['unit_price']);
        $credit_units = intval($item['credit_units']);
        $line_subtotal = $unit_price * $quantity;
        
        $stmt = $db1->prepare("
            INSERT INTO invoice_items (
                invoice_id, application_id, description, credit_units, 
                expiry_date, unit_price, quantity, line_subtotal
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $invoice_id, $item['application_id'], $item['description'], 
            $credit_units, $item['expiry_date'] ?: null, $unit_price, 
            $quantity, $line_subtotal
        ]);
        $item_id = (int)$db1->lastInsertId();
        
        $subtotal += $line_subtotal;
        
        // Add to credits ledger if invoice is being issued
        if ($issue_invoice) {
            $stmt = $db1->prepare("
                INSERT INTO credits_ledger (
                    client_id, application_id, invoice_id, invoice_item_id, credit_units, 
                    currency, expiry_date, status, created_at
                ) VALUES (?, ?, ?, ?, ?, ?, ?, 'active', CURRENT_TIMESTAMP)
            ");
            $stmt->execute([
                $client_id, $item['application_id'], $invoice_id, $item_id, 
                $credit_units, $currency, $item['expiry_date'] ?: null
            ]);
        }
    }
    
    $total = $subtotal; // No tax for now
    
    // Update invoice totals
    $stmt = $db1->prepare("
        UPDATE invoices 
        SET subtotal = ?, total = ?, 
            status = ?, 
            updated_at = CURRENT_TIMESTAMP
        WHERE id = ?
    ");
    
    $new_status = $issue_invoice ? 'issued' : 'draft';
    $stmt->execute([$subtotal, $total, $new_status, $invoice_id]);
    
    // Commit transaction
    $db1->commit();
    
    // Return success response
    $response = [
        'success' => true,
        'message' => $issue_invoice ? 'Invoice issued successfully' : 'Invoice updated successfully',
        'invoice_id' => $invoice_id,
        'redirect_url' => "?page=invoice_view&id=" . $invoice_id
    ];
    
    echo json_encode($response);
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($db1->inTransaction()) {
        $db1->rollBack();
    }
    
    error_log("Invoices Update API - Error: " . $e->getMessage());
    error_log("Invoices Update API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(400);
    echo json_encode([
        'error' => $e->getMessage(),
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
}
?>
