<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../includes/config.php';
require_once '../includes/DataBaseObjects.php';

header('Content-Type: application/json');

// Start session and check authentication
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);

try {
    $db = new DataBaseObjects();
    
    if (empty($data['payment_id']) || empty($data['allocations'])) {
        throw new Exception('Missing required fields');
    }
    
    // Normalize allocations to a consistent format: array of ['invoice_id' => int, 'amount' => float, 'currency' => string, 'units_per_usd' => float]
    $normalized_allocations = [];
    if (!is_array($data['allocations'])) {
        throw new Exception('Invalid allocations format');
    }
    foreach ($data['allocations'] as $key => $value) {
        if (is_array($value)) {
            $invoice_id = isset($value['invoice_id']) ? (int)$value['invoice_id'] : ((is_string($key) || is_int($key)) ? (int)$key : 0);
            $amount = isset($value['amount']) ? (float)$value['amount'] : (is_numeric($value) ? (float)$value : 0);
            $currency = isset($value['currency']) ? strtoupper(trim($value['currency'])) : null;
            $units = isset($value['units_per_usd']) ? (float)$value['units_per_usd'] : null;
            if ($invoice_id > 0 && $amount > 0) {
                $normalized_allocations[] = [
                    'invoice_id' => $invoice_id,
                    'amount' => $amount,
                    'currency' => $currency,
                    'units_per_usd' => $units
                ];
            }
            continue;
        }
        if ((is_string($key) || is_int($key)) && (is_numeric($value))) {
            $invoice_id = (int)$key;
            $amount = (float)$value;
            if ($invoice_id > 0 && $amount > 0) {
                $normalized_allocations[] = ['invoice_id' => $invoice_id, 'amount' => $amount];
            }
            continue;
        }
    }
    if (empty($normalized_allocations)) {
        throw new Exception('No valid allocations provided');
    }
    
    // Get payment details
    $payment = $db->select('payments', ['id' => $data['payment_id']]);
    if (empty($payment)) {
        throw new Exception('Payment not found');
    }
    $payment = $payment[0]; // Get first result
    
    // Fetch currency rates (units per USD). Expect rows like: USD=1, SAR=3.75
    $ratesRows = $db->select('currency_rates', [], '*');
    $unitsPerUsd = [];
    foreach ($ratesRows as $row) {
        $unitsPerUsd[$row['currency_code']] = (float)$row['units_per_usd'];
    }
    if (!isset($unitsPerUsd[$payment['currency']])) {
        throw new Exception('Missing currency rate for payment currency: ' . $payment['currency']);
    }

    $totalAllocatedInPaymentCurrency = 0.0;
    foreach ($normalized_allocations as $allocation) {
        // Load invoice
        $invoice = $db->select('invoices', ['id' => $allocation['invoice_id']]);
        if (empty($invoice)) {
            throw new Exception('Invoice not found: ' . $allocation['invoice_id']);
        }
        $invoice = $invoice[0];

        $invoiceCurrency = $invoice['currency'];
        if (!isset($unitsPerUsd[$invoiceCurrency])) {
            throw new Exception('Missing currency rate for invoice currency: ' . $invoiceCurrency);
        }

        // Determine allocation currency and rate; default to invoice currency if not provided
        $allocationCurrency = isset($allocation['currency']) && $allocation['currency'] ? $allocation['currency'] : $invoiceCurrency;
        $allocationUnitsPerUsd = isset($allocation['units_per_usd']) && $allocation['units_per_usd'] > 0
            ? (float)$allocation['units_per_usd']
            : (isset($unitsPerUsd[$allocationCurrency]) ? $unitsPerUsd[$allocationCurrency] : null);

        if (!$allocationUnitsPerUsd) {
            throw new Exception('Missing currency rate for allocation currency: ' . $allocationCurrency);
        }

        // Convert allocation amount to payment currency via explicit rate if provided from UI
        // If client sent 'rate_to_payment', use it directly; otherwise derive from units-per-USD mapping
        $rateToPayment = null;
        if (isset($allocation['rate_to_payment']) && is_numeric($allocation['rate_to_payment']) && (float)$allocation['rate_to_payment'] > 0) {
            $rateToPayment = (float)$allocation['rate_to_payment'];
        } else {
            $rateToPayment = $unitsPerUsd[$payment['currency']] / $allocationUnitsPerUsd;
        }
        $amountInPaymentCurrency = round((float)$allocation['amount'] * $rateToPayment, 2);
        $totalAllocatedInPaymentCurrency = round($totalAllocatedInPaymentCurrency + $amountInPaymentCurrency, 2);

        // Convert and store allocation in invoice currency for invoice balance math
        $amountInInvoiceCurrency = (float)$allocation['amount'];
        if ($allocationCurrency !== $invoiceCurrency) {
            // amount_allocation -> USD -> invoice currency
            $amountInUsd = (float)$allocation['amount'] / $allocationUnitsPerUsd; // because units per USD
            $amountInInvoiceCurrency = $amountInUsd * $unitsPerUsd[$invoiceCurrency];
        }
        $amountInInvoiceCurrency = round($amountInInvoiceCurrency, 2);

        // Record allocation
        $db->insert('payment_allocations', [
            'payment_id' => $data['payment_id'],
            'invoice_id' => $allocation['invoice_id'],
            'amount' => $amountInInvoiceCurrency,
            'allocation_currency' => $allocationCurrency,
            'allocation_units_per_usd' => $allocationUnitsPerUsd,
            'rate_to_payment' => $rateToPayment
        ]);
    }

    // Validate total does not exceed payment amount (in payment currency)
    // Include any previous allocations on this payment
    $previousAllocations = $db->select('payment_allocations', ['payment_id' => $data['payment_id']], 'SUM(amount) as total_allocated, invoice_id');
    $previousTotalInPaymentCurrency = 0.0;
    if (!empty($previousAllocations)) {
        // Prefer stored rate_to_payment when available to preserve historical conversion
        $allocRows = $db->select('payment_allocations', ['payment_id' => $data['payment_id']], '*');
        foreach ($allocRows as $row) {
            $rateToPaymentPrev = isset($row['rate_to_payment']) && is_numeric($row['rate_to_payment']) && (float)$row['rate_to_payment'] > 0
                ? (float)$row['rate_to_payment']
                : null;

            if ($rateToPaymentPrev === null) {
                // Derive from invoice currency if not stored
                $inv = $db->select('invoices', ['id' => $row['invoice_id']]);
                if (empty($inv)) { continue; }
                $invCur = $inv[0]['currency'];
                if (!isset($unitsPerUsd[$invCur])) { continue; }
                $rateToPaymentPrev = $unitsPerUsd[$payment['currency']] / $unitsPerUsd[$invCur];
            }

            $prevAmt = round(((float)$row['amount']) * $rateToPaymentPrev, 2);
            $previousTotalInPaymentCurrency = round($previousTotalInPaymentCurrency + $prevAmt, 2);
        }
    }

    if (round($previousTotalInPaymentCurrency + $totalAllocatedInPaymentCurrency - (float)$payment['amount'], 2) > 0) {
        throw new Exception('Total allocation exceeds payment amount');
    }
    
    // Update invoice statuses
    foreach ($normalized_allocations as $allocation) {
        $invoice_id = $allocation['invoice_id'];
        $allocated = $db->select('payment_allocations', ['invoice_id' => $invoice_id], 'SUM(amount) as total_allocated');
        $invoice = $db->select('invoices', ['id' => $invoice_id]);
        
        if (empty($allocated) || empty($invoice)) {
            continue; // Skip if data not found
        }
        
        $allocated = $allocated[0]; // Get first result
        $invoice = $invoice[0]; // Get first result
        
        $status = 'issued';
        if ($allocated['total_allocated'] >= $invoice['total']) {
            $status = 'paid';
        } elseif ($allocated['total_allocated'] > 0) {
            $status = 'partially_paid';
        }
        
        $db->update('invoices', ['status' => $status], ['id' => $invoice_id]);
    }
    
    echo json_encode(['success' => true]);
    
} catch (Exception $e) {
    error_log("Payments Allocate API - Error: " . $e->getMessage());
    error_log("Payments Allocate API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(400);
    echo json_encode([
        'error' => $e->getMessage(),
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
}
?>
