<?php
// Enable error reporting for debugging
error_reporting(E_ALL);
ini_set('display_errors', 1);

require_once '../includes/config.php';
require_once '../includes/DataBaseObjects.php';

header('Content-Type: application/json');

// Start session and check authentication
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Unauthorized - Admin session not found']);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    // Simple test endpoint
    echo json_encode([
        'success' => true,
        'message' => 'Payments API is working',
        'session_status' => isset($_SESSION['admin_logged_in']) ? 'active' : 'inactive',
        'admin_id' => $_SESSION['admin_id'] ?? null
    ]);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

$data = json_decode(file_get_contents('php://input'), true);

try {
    // Log incoming request
    error_log("Payments API - Incoming request: " . json_encode($data));
    
    // Check session
    if (!isset($_SESSION['admin_logged_in'])) {
        error_log("Payments API - No admin session found");
        throw new Exception('Admin session not found');
    }
    
    $db = new DataBaseObjects();
    
    if (empty($data['client_id']) || empty($data['amount']) || empty($data['currency']) || empty($data['account_id'])) {
        throw new Exception('Missing required fields: client_id, amount, currency, or account_id');
    }
    
    // Validate client exists
    $client = $db->select('tbl_clients', ['id' => $data['client_id']]);
    if (empty($client)) {
        throw new Exception('Client not found with ID: ' . $data['client_id']);
    }
    
    // Validate amount
    if (!is_numeric($data['amount']) || $data['amount'] <= 0) {
        throw new Exception('Invalid amount: must be a positive number');
    }
    
    // Validate currency
    $valid_currencies = ['USD', 'SAR', 'EGP', 'EUR', 'GBP'];
    if (!in_array($data['currency'], $valid_currencies)) {
        throw new Exception('Invalid currency: must be one of ' . implode(', ', $valid_currencies));
    }
    
    // Validate account exists and is active
    $account = $db->select('accounts', ['id' => $data['account_id']], '*');
    if (empty($account) || (isset($account[0]['is_active']) && (int)$account[0]['is_active'] !== 1)) {
        throw new Exception('Account not found or inactive with ID: ' . $data['account_id']);
    }

    // Prepare payment data
    $payment_data = [
        'client_id' => $data['client_id'],
        'payment_date' => $data['payment_date'] ?? date('Y-m-d'),
        'amount' => $data['amount'],
        'currency' => $data['currency'],
        'method' => $data['method'] ?? 'other',
        'account_id' => $data['account_id'],
        'reference' => $data['reference'] ?? null,
        'notes' => $data['notes'] ?? null,
        'created_by' => $_SESSION['admin_id'] ?? null,
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    error_log("Payments API - Inserting payment: " . json_encode($payment_data));
    
    $payment_id = $db->insert('payments', $payment_data);
    
    if (!$payment_id) {
        throw new Exception('Failed to insert payment into database');
    }
    
    error_log("Payments API - Payment created successfully with ID: " . $payment_id);
    
    // Insert into account ledger (credit the account)
    $ledger_entry = [
        'account_id' => $data['account_id'],
        'entry_date' => $data['payment_date'] ?? date('Y-m-d'),
        'type' => 'credit',
        'amount' => $data['amount'],
        'currency' => $data['currency'],
        'source' => 'payment',
        'source_id' => $payment_id,
        'description' => isset($data['reference']) && $data['reference'] ? ('Payment ' . $data['reference']) : 'Payment received',
        'created_by' => $_SESSION['admin_id'] ?? null,
        'created_at' => date('Y-m-d H:i:s')
    ];

    $db->insert('account_ledger', $ledger_entry);

    echo json_encode([
        'success' => true,
        'payment_id' => $payment_id,
        'message' => 'Payment created successfully'
    ]);
    
} catch (Exception $e) {
    error_log("Payments API - Error: " . $e->getMessage());
    error_log("Payments API - Stack trace: " . $e->getTraceAsString());
    
    http_response_code(400);
    echo json_encode([
        'error' => $e->getMessage(),
        'debug_info' => DEVELOPMENT_MODE ? [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ] : null
    ]);
}
?>
