<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get invoice ID from URL
$invoice_id = $_GET['id'] ?? 0;

if (!$invoice_id) {
    echo '<div class="alert alert-danger">Invoice ID is required</div>';
    return;
}

// Get invoice details
$invoice = null;
$invoice_items = [];
$payment_allocations = [];

try {
    // Get invoice header
    $stmt = $db1->prepare("
        SELECT i.*, c.CompanyName as client_name
        FROM invoices i
        LEFT JOIN tbl_clients c ON i.client_id = c.Id
        WHERE i.id = ?
    ");
    $stmt->execute([$invoice_id]);
    $invoice = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$invoice) {
        echo '<div class="alert alert-danger">Invoice not found</div>';
        return;
    }
    
    // Get invoice items
    $stmt = $db1->prepare("
        SELECT ii.*, a.Application_Name
        FROM invoice_items ii
        LEFT JOIN tbl_applications a ON ii.application_id = a.id
        WHERE ii.invoice_id = ?
        ORDER BY ii.id
    ");
    $stmt->execute([$invoice_id]);
    $invoice_items = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get payment allocations
    $stmt = $db1->prepare("
        SELECT pa.*, p.payment_date, p.method, p.reference
        FROM payment_allocations pa
        LEFT JOIN payments p ON pa.payment_id = p.id
        WHERE pa.invoice_id = ?
        ORDER BY pa.created_at
    ");
    $stmt->execute([$invoice_id]);
    $payment_allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    echo '<div class="alert alert-danger">Error loading invoice: ' . htmlspecialchars($e->getMessage()) . '</div>';
    return;
}
?>

<!-- Invoice View Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Invoice #<?= htmlspecialchars($invoice['invoice_number']) ?></h2>
    <div>
        <a href="?page=invoices" class="btn btn-secondary">Back to Invoices</a>
        <?php if ($invoice['status'] == 'draft'): ?>
            <a href="?page=invoice_edit&id=<?= $invoice['id'] ?>" class="btn btn-warning">Edit Invoice</a>
        <?php endif; ?>
    </div>
</div>

<!-- Invoice Header -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Invoice Details</h5>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <p><strong>Client:</strong> <?= htmlspecialchars($invoice['client_name']) ?></p>
                <p><strong>Invoice Number:</strong> <?= htmlspecialchars($invoice['invoice_number']) ?></p>
                <p><strong>Issue Date:</strong> <?= $invoice['issue_date'] ?></p>
                <p><strong>Due Date:</strong> <?= $invoice['due_date'] ?: 'Not specified' ?></p>
            </div>
            <div class="col-md-6">
                <p><strong>Status:</strong> 
                    <span class="badge bg-<?= getStatusColor($invoice['status']) ?>">
                        <?= ucfirst(str_replace('_', ' ', $invoice['status'])) ?>
                    </span>
                </p>
                <p><strong>Currency:</strong> <?= htmlspecialchars($invoice['currency']) ?></p>
                <p><strong>Subtotal:</strong> <?= number_format($invoice['subtotal'], 2) ?></p>
                <p><strong>Total:</strong> <?= number_format($invoice['total'], 2) ?></p>
            </div>
        </div>
        <?php if ($invoice['notes']): ?>
            <div class="mt-3">
                <strong>Notes:</strong>
                <p class="mb-0"><?= nl2br(htmlspecialchars($invoice['notes'])) ?></p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Invoice Items -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Invoice Items</h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Application</th>
                        <th>Description</th>
                        <th>Credit Units</th>
                        <th>Expiry Date</th>
                        <th>Unit Price</th>
                        <th>Quantity</th>
                        <th>Subtotal</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($invoice_items)): ?>
                        <tr>
                            <td colspan="7" class="text-center">No items found</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($invoice_items as $item): ?>
                            <tr>
                                <td><?= htmlspecialchars($item['Application_Name'] ?? 'Unknown') ?></td>
                                <td><?= htmlspecialchars($item['description']) ?></td>
                                <td><?= $item['credit_units'] ?></td>
                                <td><?= $item['expiry_date'] ?: 'No expiry' ?></td>
                                <td><?= number_format($item['unit_price'], 2) ?></td>
                                <td><?= $item['quantity'] ?></td>
                                <td><?= number_format($item['line_subtotal'], 2) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>

<!-- Payment Allocations -->
<?php if (!empty($payment_allocations)): ?>
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Payment Allocations</h5>
    </div>
    <div class="card-body">
        <div class="table-responsive">
            <table class="table table-striped">
                <thead>
                    <tr>
                        <th>Payment Date</th>
                        <th>Method</th>
                        <th>Reference</th>
                        <th>Amount Allocated</th>
                        <th>Date Allocated</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($payment_allocations as $allocation): ?>
                        <tr>
                            <td><?= $allocation['payment_date'] ?></td>
                            <td>
                                <span class="badge bg-info">
                                    <?= ucfirst($allocation['method']) ?>
                                </span>
                            </td>
                            <td><?= htmlspecialchars($allocation['reference'] ?: '-') ?></td>
                            <td><?= number_format($allocation['amount'], 2) ?></td>
                            <td><?= $allocation['created_at'] ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    </div>
</div>
<?php endif; ?>

<?php
function getStatusColor($status) {
    switch ($status) {
        case 'draft': return 'secondary';
        case 'issued': return 'primary';
        case 'partially_paid': return 'warning';
        case 'paid': return 'success';
        case 'void': return 'danger';
        default: return 'secondary';
    }
}
?>
