<?php
// Check if user is logged in
if (!isset($_SESSION['admin_logged_in'])) {
    header('Location: ../login.php');
    exit;
}

// Get payment ID from URL
$payment_id = $_GET['id'] ?? 0;

if (!$payment_id) {
    echo '<div class="alert alert-danger">Payment ID is required</div>';
    return;
}

// Get payment details
$payment = null;
$payment_allocations = [];

try {
    // Get payment header
    $stmt = $db1->prepare("
        SELECT p.*, c.CompanyName as client_name
        FROM payments p
        LEFT JOIN tbl_clients c ON p.client_id = c.Id
        WHERE p.id = ?
    ");
    $stmt->execute([$payment_id]);
    $payment = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$payment) {
        echo '<div class="alert alert-danger">Payment not found</div>';
        return;
    }
    
    // Get payment allocations
    $stmt = $db1->prepare("
        SELECT pa.*, i.invoice_number, i.issue_date, i.total as invoice_total
        FROM payment_allocations pa
        LEFT JOIN invoices i ON pa.invoice_id = i.id
        WHERE pa.payment_id = ?
        ORDER BY pa.created_at
    ");
    $stmt->execute([$payment_id]);
    $payment_allocations = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    echo '<div class="alert alert-danger">Error loading payment: ' . htmlspecialchars($e->getMessage()) . '</div>';
    return;
}

// Calculate totals in payment currency using stored rate_to_payment when available
$allocated_amount = 0.0;
try {
    // Load currency rates (units per USD)
    $rates = [];
    $rs = $db1->query("SELECT currency_code, units_per_usd FROM currency_rates");
    if ($rs) {
        while ($r = $rs->fetch(PDO::FETCH_ASSOC)) { $rates[$r['currency_code']] = (float)$r['units_per_usd']; }
    }

    $payUnits = isset($rates[$payment['currency']]) ? (float)$rates[$payment['currency']] : null;

    foreach ($payment_allocations as $row) {
        $rateToPayment = isset($row['rate_to_payment']) && is_numeric($row['rate_to_payment']) && (float)$row['rate_to_payment'] > 0
            ? (float)$row['rate_to_payment']
            : null;

        if ($rateToPayment === null) {
            // derive from invoice currency if available in join; we don't have it, fetch quickly
            $invCurStmt = $db1->prepare("SELECT currency FROM invoices WHERE id = ?");
            $invCurStmt->execute([$row['invoice_id']]);
            $invRow = $invCurStmt->fetch(PDO::FETCH_ASSOC);
            if ($invRow && isset($rates[$invRow['currency']]) && $payUnits) {
                $rateToPayment = $payUnits / (float)$rates[$invRow['currency']];
            }
        }

        if ($rateToPayment) {
            $allocated_amount = round($allocated_amount + round(((float)$row['amount']) * $rateToPayment, 2), 2);
        }
    }
} catch (Exception $e) {
    $allocated_amount = 0.0;
}

$remaining_amount = round((float)$payment['amount'] - $allocated_amount, 2);
?>

<!-- Payment View Content -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2>Payment Details</h2>
    <div>
        <a href="?page=payments" class="btn btn-secondary">Back to Payments</a>
        <?php if ($remaining_amount > 0): ?>
            <a href="?page=payment_allocate&id=<?= $payment['id'] ?>" class="btn btn-warning">Allocate Payment</a>
        <?php endif; ?>
    </div>
</div>

<!-- Payment Header -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Payment Information</h5>
    </div>
    <div class="card-body">
        <div class="row">
            <div class="col-md-6">
                <p><strong>Client:</strong> <?= htmlspecialchars($payment['client_name']) ?></p>
                <p><strong>Payment Date:</strong> <?= $payment['payment_date'] ?></p>
                <p><strong>Amount:</strong> <?= number_format($payment['amount'], 2) ?></p>
                <p><strong>Currency:</strong> <?= htmlspecialchars($payment['currency']) ?></p>
            </div>
            <div class="col-md-6">
                <p><strong>Method:</strong> 
                    <span class="badge bg-info">
                        <?= ucfirst($payment['method']) ?>
                    </span>
                </p>
                <p><strong>Reference:</strong> <?= htmlspecialchars($payment['reference'] ?: 'Not specified') ?></p>
                <p><strong>Allocated:</strong> <?= number_format($allocated_amount, 2) ?></p>
                <p><strong>Remaining:</strong> 
                    <span class="badge bg-<?= $remaining_amount > 0 ? 'warning' : 'success' ?>">
                        <?= number_format($remaining_amount, 2) ?>
                    </span>
                </p>
            </div>
        </div>
        <?php if ($payment['notes']): ?>
            <div class="mt-3">
                <strong>Notes:</strong>
                <p class="mb-0"><?= nl2br(htmlspecialchars($payment['notes'])) ?></p>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Payment Allocations -->
<div class="card mb-4">
    <div class="card-header">
        <h5 class="mb-0">Payment Allocations</h5>
    </div>
    <div class="card-body">
        <?php if (empty($payment_allocations)): ?>
            <div class="alert alert-info">
                No allocations have been made for this payment yet.
                <?php if ($remaining_amount > 0): ?>
                    <a href="?page=payment_allocate&id=<?= $payment['id'] ?>" class="btn btn-sm btn-warning ms-2">Allocate Now</a>
                <?php endif; ?>
            </div>
        <?php else: ?>
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Invoice #</th>
                            <th>Issue Date</th>
                            <th>Invoice Total</th>
                            <th>Amount Allocated</th>
                            <th>Date Allocated</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($payment_allocations as $allocation): ?>
                            <tr>
                                <td>
                                    <a href="?page=invoice_view&id=<?= $allocation['invoice_id'] ?>">
                                        <?= htmlspecialchars($allocation['invoice_number']) ?>
                                    </a>
                                </td>
                                <td><?= $allocation['issue_date'] ?></td>
                                <td><?= number_format($allocation['invoice_total'], 2) ?></td>
                                <td><?= number_format($allocation['amount'], 2) ?></td>
                                <td><?= $allocation['created_at'] ?></td>
                                <td>
                                    <a href="?page=invoice_view&id=<?= $allocation['invoice_id'] ?>" class="btn btn-sm btn-info">View Invoice</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                    <tfoot>
                        <tr class="table-info">
                            <td colspan="3"><strong>Total Allocated:</strong></td>
                            <td><strong><?= number_format($allocated_amount, 2) ?></strong></td>
                            <td colspan="2"></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Summary -->
<div class="card">
    <div class="card-body">
        <div class="row">
            <div class="col-md-4">
                <div class="text-center">
                    <h4><?= number_format($payment['amount'], 2) ?></h4>
                    <p class="text-muted">Total Payment Amount</p>
                </div>
            </div>
            <div class="col-md-4">
                <div class="text-center">
                    <h4 class="text-success"><?= number_format($allocated_amount, 2) ?></h4>
                    <p class="text-muted">Amount Allocated</p>
                </div>
            </div>
            <div class="col-md-4">
                <div class="text-center">
                    <h4 class="text-<?= $remaining_amount > 0 ? 'warning' : 'success' ?>"><?= number_format($remaining_amount, 2) ?></h4>
                    <p class="text-muted">Amount Remaining</p>
                </div>
            </div>
        </div>
    </div>
</div>
